# query.reduce()

Iterates over every row of the given query and calls the closure with that row.

This function will reduce the query to a single value and will return this value.

```
query.reduce( closure=function, initialValue=object )
```

**Returns:** any

# Arguments

| Argument | Type | Required | Description | Default |
|----------|------|----------|-------------|---------|
| closure | function | Yes | function/closure that implements the following constructor `function(any result, struct row[, number rowNumber, query query]){ return result;}`. *Alias: function, callback, udf* |  |
| initialValue | object | No | initial value passed as part of the first closure call *Alias: initial, initalValue* |  |

# Examples

```cfml
	people = QueryNew(
	"name,dob,age", "varchar,date,int",
	[
		[ "Susi", CreateDate( 1970, 1, 1 ), 0 ],
		[ "Urs" , CreateDate( 1995, 1, 1 ), 0 ],
		[ "Fred", CreateDate( 1960, 1, 1 ), 0 ],
		[ "Jim" , CreateDate( 1988, 1, 1 ), 0 ]
	]
	);
	dump( var=people, label="people - original query" );
	/* Output:
	| name | dob                 | age |
	------------------------------------
	| Susi | 1970-01-01 00:00:00 | 0   |
	| Urs  | 1995-01-01 00:00:00 | 0   |
	| Fred | 1960-01-01 00:00:00 | 0   |
	| Jim  | 1988-01-01 00:00:00 | 0   |
	*/

	// initialise age as a second argument
	totalAge = people.reduce( function(age, row, rowNumber, recordset ){
	    return age +  DateDiff( 'yyyy', recordset.dob, Now() );
	},0);
	Dump( var=totalAge, label='people - total age' );

	/* Output:
	    totalAge = 167
	*/

	// initialise age in closure
	totalAge = people.reduce( function(age=0, row, rowNumber, recordset ){
	    return age +  DateDiff( 'yyyy', recordset.dob, Now() );
	});

	// Here you can get NULL value if there is no recordset
	if( isNull(totalAge) ) {
	    totalAge = 0;
	}
	dump( var=totalAge, label='people - total age' );

	/* Output: if recordset exists else 0
	      totalAge = 167
	*/
```

# Categories

[Iterator](../../../categories/iterator.md), [Queries](../../../categories/query.md), [Threads](../../../categories/thread.md)

# See Also

[QueryReduce()](../../functions/queryreduce.md), [Query](../query.md)