# QueryFilter()

This function creates a new Query that returns all rows from an query that match the given filter.

```
QueryFilter( query=query, filter=function, parallel=boolean, maxThreads=number );
```

**Returns:** query

# Arguments

| Argument | Type | Required | Description | Default |
|----------|------|----------|-------------|---------|
| query | query | Yes | query to filter entries from |  |
| filter | function | Yes | A function/closure that implements the following constructor `function(struct row[, number rowNumber, query query]) { return true/false; }` for best performance, use scoped variables, like `arguments.row`, instead of just `row` |  |
| parallel | boolean | No | execute closures parallel |  |
| maxThreads | number | No | maximum number of threads executed, ignored when argument "parallel" is set to false *Alias: maxThreadCount* | 20 |

# Examples

```cfml
people = QueryNew( "name,dob", "varchar,date", [
    [ "Susi", CreateDate( 1970, 1, 1 ), 0 ],
    [ "Urs" , CreateDate( 1995, 1, 1 ), 0 ],
    [ "Fred", CreateDate( 1960, 1, 1 ), 0 ],
    [ "Jim" , CreateDate( 1988, 1, 1 ), 0 ],
    [ "Bob" , CreateDate( 1988, 1, 1 ), 0 ]
]);

Dump( var=people, label="people - original query" );

/* Output:

| name | dob                 |
------------------------------
| Susi | 1970-01-01 00:00:00 |
| Urs  | 1995-01-01 00:00:00 |
| Fred | 1960-01-01 00:00:00 |
| Jim  | 1988-01-01 00:00:00 |
| Bob  | 1988-01-01 00:00:00 |
*/

// filter - born in 1988
qryPeopleBornIn1988 = people.filter(function(row, rowNumber, qryData){
    return Year( row.dob ) == 1988
});

dump(var=qryPeopleBornIn1988, label='qryPeopleBornIn1988 - Born in 1988');

/* Output:

| name | dob                 |
------------------------------
| Jim  | 1988-01-01 00:00:00 |
| Bob  | 1988-01-01 00:00:00 |
*/
```

```cfml
<cfscript>
	q = QueryNew("name, description");
	loop times=3 {
		getFunctionList().each(function(f){
			var fd = getFunctionData(arguments.f);
			var r =QueryAddRow(q);
			QuerySetCell(q,"name", fd.name, r);
			QuerySetCell(q,"description", fd.description, r);
		});
	}
	dump(var=q.recordcount,
	    label="demo data set size");
	s = "the";
</cfscript>

<cftimer type="outline" label="Query of Query">
	<cfquery dbtype="query" name="q1">
		select 	name, description
		from 	q
		where 	description like <cfqueryparam value='%#s#%' cfsqltype="varchar">
	</cfquery>
</cftimer>
<cfdump var=#q1.recordcount#>

<cftimer type="outline" label="query.filter() with scoped variables">
	<cfscript>
		q2 = q.filter(function(row){
			return (arguments.row.description contains s);
		});
	</cfscript>
</cftimer>
<cfdump var=#q2.recordcount#>

<cftimer type="outline" label="query.filter() without unscoped variables">
	<cfscript>
		q3 = q.filter(function(row){
			return (row.description contains s);
		});
	</cfscript>
</cftimer>
<cfdump var=#q3.recordcount#>

```







# Categories

[Collections](../../categories/collection.md), [Queries](../../categories/query.md), [Threads](../../categories/thread.md)

# See Also

[query.filter()](../objects/query/filter.md), [Query of Queries sometimes it rocks, sometimes it sucks](../../recipes/query-of-query-sometime.md)