# QueryEach()

call the given UDF/Closure with every row (struct) in the query.

```
QueryEach( query=query, closure=function, parallel=boolean, maxThreads=number );
```

**Returns:** void

# Arguments

| Argument | Type | Required | Description | Default |
|----------|------|----------|-------------|---------|
| query | query | Yes | query to take values from |  |
| closure | function | Yes | A closure function (UDF) with the following signature: `function( row, rowNumber, recordset ){}` Which is called for each row in the query. *Alias: function, callback, udf* |  |
| parallel | boolean | No | execute closures parallel |  |
| maxThreads | number | No | maximum number of threads executed, ignored when argument "parallel" is set to false *Alias: maxThreadCount* | 20 |

# Examples

```cfml
people = QueryNew( "name,dob,age", "varchar,date,int", [
    [ "Susi", CreateDate( 1970, 1, 1 ), 0 ],
    [ "Urs" , CreateDate( 1995, 1, 1 ), 0 ],
    [ "Fred", CreateDate( 1960, 1, 1 ), 0 ],
    [ "Jim" , CreateDate( 1988, 1, 1 ), 0 ]
]);

Dump( var=people, label="people - original query" );

/* Output:

| name | dob                 | age |
------------------------------------
| Susi | 1970-01-01 00:00:00 | 0   |
| Urs  | 1995-01-01 00:00:00 | 0   |
| Fred | 1960-01-01 00:00:00 | 0   |
| Jim  | 1988-01-01 00:00:00 | 0   |

*/

people.each( function( row, rowNumber, recordset ){
    recordset.age[ rowNumber ] = DateDiff( 'yyyy', row.dob, Now() );
});

Dump( var=people, label='people - with calculated age' );

/* Output:

| name | dob                 | age |
------------------------------------
| Susi | 1970-01-01 00:00:00 | 45  |
| Urs  | 1995-01-01 00:00:00 | 20  |
| Fred | 1960-01-01 00:00:00 | 55  |
| Jim  | 1988-01-01 00:00:00 | 27  |

*/
```







# Categories

[Collections](../../categories/collection.md), [Queries](../../categories/query.md), [Threads](../../categories/thread.md)

# See Also

[query.each()](../objects/query/each.md)