# Hash()

Returns the one-way hash of a variable-length input string based on the algorithm, producing a fixed-length string.

A one-way hash cannot be reversed back to the original input except by a brute force attack.

```
Hash( input=any, algorithm=string, encoding=string, numIterations=number );
```

**Returns:** string

# Arguments

| Argument | Type | Required | Description | Default |
|----------|------|----------|-------------|---------|
| input | any | Yes | String to hash. *Alias: string* |  |
| algorithm | string | No | The algorithm to use for hashing your input. The following values are supported: * QUICK: Generates a 15 or 16-character, hexadecimal string. This algorithm provides fast hashing but offers no cryptographic security. * MD5: Generates a 32-character string using the MD5 algorithm. This algorithm is considered cryptographically broken and unsuitable for further cryptographical use. * CFMX_COMPAT: Deprecated. Lucee uses the MD5 algorithm instead. * SHA: Generates a 40-character string using the Secure Hash Algorithm, SHA-1, specified by National Institute of Standards and Technology (NIST) in the FIPS-180-2 standard. Cryptographic weaknesses were discovered in SHA-1, and the standard is no longer approved for most cryptographic uses. * SHA-256: Generates a 64-character string using the SHA-256 algorithm specified by the FIPS-180-2 standard. * SHA-384: Generates a 96-character string using the SHA-384 algorithm specified by the FIPS-180-2 standard. * SHA-512: Generates an 128-character string using the SHA-512 algorithm specified by the FIPS-180-2 standard. The default value is MD5. |  |
| encoding | string | No | The encoding to use when converting the input string to byte data. |  |
| numIterations | number | No | The number of times to iterate hashing. A greater number of iterations will result in a cryptographically stronger hash but will also take more time to generate that hash. The default value is 1. |  |

# Examples

```cfml
// Variable-length strings produce fixed-length hashes.
input = 'Hello World';
dump( hash( input ) ); // B10A8DB164E0754105B7A99BE72E3FE5

input = 'Mary had a little lamb.';
dump( hash( input ) ); // CA964B1677D5476EA11EED1E1837C342
```

```cfml
// Different algorithms are supported.
input = 'Hello World';
dump( hash( input ) ); // B10A8DB164E0754105B7A99BE72E3FE5
dump( hash( input, 'QUICK' ) ); // 6b736ba1c9a95606
dump( hash( input, 'MD5' ) ); // B10A8DB164E0754105B7A99BE72E3FE5
dump( hash( input, 'CFMX_COMPAT' ) ); // B10A8DB164E0754105B7A99BE72E3FE5
dump( hash( input, 'SHA' ) ); // 0A4D55A8D778E5022FAB701977C5D840BBC486D0
dump( hash( input, 'SHA-256' ) ); // A591A6D40BF420404A011733CFB7B190D62C65BF0BCDA32B57B277D9AD9F146E
dump( hash( input, 'SHA-384' ) ); // 99514329186B2F6AE4A1329E7EE6C610A729636335174AC6B740F9028396FCC803D0E93863A7C3D90F86BEEE782F4F3F
dump( hash( input, 'SHA-512' ) ); // 2C74FD17EDAFD80E8447B0D46741EE243B7EB74DD2149A0AB1B9246FB30382F27E853D8585719E0E67CBDA0DAA8F51671064615D645AE27ACB15BFB1447F459B

// The default, MD5 and CFMX_COMPAT algorithms are all MD5.
dump( hash( input ) == hash( input, 'MD5' ) && hash( input, 'MD5' ) == hash( input, 'CFMX_COMPAT' ) ); // true
```

```cfml
input = 'Hello World';

// The number of iterations change the hash
dump( hash( input = input, numIterations = 1 ) ); // B10A8DB164E0754105B7A99BE72E3FE5
dump( hash( input = input, numIterations = 9 ) ); // 5E1C304FD939BBE1378ED977E2AD26B5

// numIterations less than 1 are set to 1
dump( hash( input = input, numIterations = 0 ) == hash( input = input, numIterations = 1 ) ); // true

// More iterations take more time
timer type='inline' label='1000 SHA-512s, 1 iteration each' {
    for ( i = 0; i < 1000; i++ ) {
        hash( input = input, algorithm = 'SHA-512', numIterations = 1 );
    }
    // 1000 SHA-512s, 1 iteration each: 5ms
};

writeOutput('<br>');

timer type='inline' label='1000 SHA-512s, 10 iterations each' {
    for ( i = 0; i < 1000; i++ ) {
        hash( input = input, algorithm = 'SHA-512', numIterations = 10 );
    }
    // 1000 SHA-512s, 10 iterations each: 21ms
};
```







# Categories

[Cryptography](../../categories/crypto.md)

# See Also

[Hash40()](hash40.md), [string.hash()](../objects/string/hash.md)