# CreateObject()

Creates and returns a reference to objects of various types including Java classes, components, web services, and COM objects.

## Type: "java"

Creates a Java object instance. As of Lucee 6.2, this supports Maven dependency management and enhanced Java settings.

```
CreateObject('java', className [, classpath|bundleName|javasettings] [, delimiter|bundleVersion])
```

You can provide Java libraries in four ways:

- Using classpath: Specify directories or JAR files
- Using OSGi bundles: Provide a bundle name and version
- Using loadPaths: Define non-OSGi JAR paths directly
- Using Maven: Define Maven artifacts via a javasettings structure

### Using Classpath

```cfml
// Using directories and JAR files with a classpath
createObject("java", "com.example.MyClass", "/path/to/libs/,/path/to/specific.jar");

// Using an array of paths
createObject("java", "com.example.MyClass", ["/path/to/libs/", "/path/to/specific.jar"]);
```

### OSGi Bundle Support

```cfml
createObject(
  type: "java", 
  class: "org.lucee.mockup.osgi.Test", 
  bundleName: "lucee.mockup", 
  bundleVersion: "1.0.0.0"
);
```

### Maven Support (Lucee 6.2+)

```cfml
createObject(
  type: "java",
  class: "org.apache.commons.beanutils.BeanUtils",
  javasettings: {
    "maven": [
      {
        "groupId": "commons-beanutils",
        "artifactId": "commons-beanutils",
        "version": "1.9.4"
      }
    ]
  }
);
```

### Java Settings Structure

The javasettings structure can contain:

- maven: Array of Maven dependency structures (groupId, artifactId, version)
- loadPaths: Array of paths to non-OSGi JAR files or directories
- bundlePaths: Array of paths to OSGi bundle files or directories
- reloadOnChange: Boolean indicating whether to reload updated classes dynamically (default: false)
- watchInterval: Number of seconds between checks for changes (default: 60)
- watchExtensions: Array of file extensions to monitor (default: ["jar", "class"])

## Type: "component"

Creates a CFC instance without calling the init method.

```
CreateObject('component', componentName)
```

## Type: "webservice"

Creates a web service client from a WSDL.

```
CreateObject('webservice', urlToWsdl [, portName] [, options])
```

## Type: "com"

Creates a COM object (Windows only).

```
CreateObject('com', class, context, serverName)
```

```
CreateObject( type=string, classname=object, context=object, delimiterOrVersion=object );
```

**Returns:** any

# Arguments

| Argument | Type | Required | Description | Default |
|----------|------|----------|-------------|---------|
| type | string | Yes | one of the following: - com: for loading a com Object - java: for loading a Java object - webservice: for loading a remote webservice - component: for loading a Component *Alias: componentType* |  |
| classname | object | No | the usage of this argument depend on type defined with argument "type": - com: Component ProgID for the object to invoke. - java: Java class to load - webservice: WSDL url to call - component: The CFC name; corresponds to the name of the file that defines the component *Alias: component, class* |  |
| context | object | No | The usage of this argument varies, depending on the value of the `type` argument: - "java": * Classpath: String list or array of paths to class files or JAR files * BundleName: Name of an OSGi bundle * JavaSettings: Structure containing Maven and Java configuration including: - maven: Array of Maven dependencies (groupId, artifactId, version) - loadPaths: Array of paths to non-OSGi JAR files or directories - bundlePaths: Array of paths to OSGi bundle files or directories - reloadOnChange: Boolean to enable dynamic reload of updated classes - watchInterval: Interval in seconds to check for changes - watchExtensions: File extensions to monitor - "webservice": Structure with web service options (username, password, proxyServer, proxyPort, proxyUser, proxyPassword) - "component": Not used - "com": Not used *Alias: bundleName, name, webservice, Assembly, WSPortName, thirdArg, javasettings* |  |
| delimiterOrVersion | object | No | the usage of this argument depend on type defined with argument "type": - java: delimiter used for the classpath (default comma) or bundle version when the previous argument was a bundle name - webservice: not used (ignored) - component: not used (ignored) - com: not used (ignored) *Alias: delimiter, version, bundleVersion* |  |

# Examples

```cfml
dump( label:"CreateObject in java",  var:createObject('java','java.util.HashMap'));
dump( label:"CreateObject with init()", var:createObject('java',"java.lang.StringBuffer").init());
```

### All these examples are all functionally identical

*Except that the `new Object()` syntax automatically calls the `init()` method*

```cfml
dump( var=createObject("component", "org.lucee.cfml.http"), expand=false );
// but even "component" is optional for cfcs
dump( var=createObject("org.lucee.cfml.http"), expand=false );

// the modern new Object() syntax is also dynamic 
dump(var=new "org.lucee.cfml.http"(), expand=false);
dump(var=new org.lucee.cfml.http(), expand=false);

cfc = "org.lucee.cfml.http";
dump(var=new "#cfc#"(), expand=false);
```







# Categories

[Components (CFCs)](../../categories/component.md), [Core CFML Language](../../categories/core.md), [Java](../../categories/java.md), [Objects](../../categories/object.md), [Web services](../../categories/webservice.md)

# See Also

[MavenLoad()](mavenload.md), [Import](../../recipes/import.md), [Java Class Interaction](../../recipes/java-class-interaction.md), [JavaSettings in Application.cfc, Components and CFConfig.json](../../recipes/java-settings.md), [Using Maven directly via CFML](../../recipes/maven.md), [New Operator in Lucee](../../recipes/new-operator.md), [<cfimport>](../tags/import.md)