# AI Session Serialization

**Introduced:** 6.0



# AI Session Serialization

Lucee provides functions to serialize and deserialize AI sessions, allowing you to save conversation state and restore it later. 

This enables persistent AI conversations across page requests, application restarts, or sharing conversation contexts between different parts of your application.

## Functions

Two key functions handle AI session serialization:

### SerializeAISession

Converts an AI session object to a JSON string containing configuration settings and conversation history.

```cfml
SerializeAISession(session, maxlength, condense=false)
```

**Arguments:**

- `session`: The AI session object returned by LuceeCreateAISession(). This session maintains the conversation history and configuration settings like temperature and system message.
- `maxlength`: Optional. Specifies the maximum number of conversation exchanges to retain in the serialized output. When specified, the function preserves the most recent exchanges up to this limit, truncating older exchanges while keeping the session configuration intact. This helps control memory usage and storage requirements for long-running conversations without losing recent context. When used with condense=true, maxlength is applied before condensation. If not specified, all conversation history is retained.
- `condense`: Controls whether the conversation history should be condensed before serialization. When set to true, the function will intelligently summarize older parts of the conversation while preserving essential context. This reduces the overall token count and storage requirements while maintaining the important elements of the conversation history. Default is false, which preserves the complete conversation history.

**Returns:**

A JSON string containing:

- Configuration settings (temperature, limits, timeouts, system message)
- Conversation history with questions and answers, which may be:
  - Complete history (if no parameters are used)
  - Truncated history (when maxlength is specified)
  - Condensed history (when condense=true)
  - First truncated, then condensed (when both parameters are used)

### LoadAISession

Recreates an AI session from previously serialized data.

```cfml
LoadAISession(name, data)
```

**Arguments:**

- `name`: Specifies which AI endpoint configuration to use. Can be:
  - Direct endpoint name (e.g., 'claude')
  - ID-based reference with prefix (e.g., 'id:123')
  - Default category reference (e.g., 'default:exception')
- `data`: The serialized AI session data, either as:
  - A JSON string previously generated by SerializeAISession()
  - A struct containing session configuration and conversation history

**Returns:**

A new AI session object with the restored configuration and conversation history.

## Cross-Model Flexibility

An important feature of these functions is that you are not bound to a specific endpoint when loading a session. 

You can serialize a conversation from one AI model and load it into a different one. For example:

- Serialize a conversation with Claude
- Load that conversation state into ChatGPT or Gemini
- Continue the conversation with the new model while maintaining the conversation history

This allows for:

- A/B testing different models with the same conversation history
- Falling back to alternative models if one is unavailable
- Moving conversations to more capable or cost-effective models as needed

## Usage Examples

### Basic Serialization and Restoration

```cfml
// Create and use an AI session
claudeSession = LuceeCreateAISession(name:'claude', systemMessage:"Answer as succinctly as possible.");
response = LuceeInquiryAISession(claudeSession, "What is the capital of France?");

// Serialize session to JSON string
serializedData = SerializeAISession(claudeSession);

// Save to database, file, etc.
fileWrite(myAISession, serializedData);

// ... Later, in another request ...

// Restore the session with complete history
// Note that we're using 'chatGPT' here instead of 'claude'
serializedData = fileRead(myAISession);
restoredSession = LoadAISession('chatGPT', serializedData);

// Continue conversation with context preserved, but now using ChatGPT
response = LuceeInquiryAISession(restoredSession, "What is the capital of Switzerland?");
```

### Limiting Conversation History Length

```cfml
// After a lengthy conversation with many exchanges
claudeSession = LuceeCreateAISession(name:'claude');
// ... 30+ interactions ...

// Serialize but keep only the 10 most recent exchanges
serializedData = SerializeAISession(session:claudeSession, maxlength:10);

// Save the trimmed session
fileWrite(myAISession, serializedData);

// Restore later with only the 10 most recent exchanges
restoredSession = LoadAISession('claude', fileRead(myAISession));
```

### Condensing Conversation History

```cfml
// After a lengthy conversation with many exchanges
claudeSession = LuceeCreateAISession(name:'claude');
// ... many interactions ...

// Serialize with condensed history to reduce token count
serializedData = SerializeAISession(session:claudeSession, condense:true);

// Save the condensed session
fileWrite(myAISession, serializedData);

// Restore later with the condensed but contextually complete history
restoredSession = LoadAISession('claude', fileRead(myAISession));

// Continue conversation with preserved essential context
response = LuceeInquiryAISession(restoredSession, "Can you summarize what we discussed?");
```

### Combined Limiting and Condensing

```cfml
// For very long conversations, combine both approaches
// Note: maxlength is applied BEFORE condensation
serializedData = SerializeAISession(session:claudeSession, maxlength:20, condense:true);

// This means that first, the conversation is limited to the 20 most recent exchanges
// Then, those remaining exchanges are condensed/summarized
```

## Security Considerations

- The serialized data contains the conversation history (complete or partial), including any sensitive information shared with the AI
- Consider encryption if storing the serialized data in external systems

## Future Development

The serialization functions will continue to evolve alongside Lucee's AI capabilities:

- Support for multi-part requests and responses
- Compression options for large conversation histories
- Additional metadata and tagging options

# Categories

[AI / LLM Integration](../categories/ai.md)